// FocusOps Quick Add - Popup Script

// DOM Elements for Auth
const authChecking = document.getElementById('authChecking');
const authRequired = document.getElementById('authRequired');
const loginBtn = document.getElementById('loginBtn');

// DOM Elements for Settings
const settingsPanel = document.getElementById('settingsPanel');
const settingsAppUrl = document.getElementById('settingsAppUrl');
const openSettingsBtn = document.getElementById('openSettingsBtn');
const backBtn = document.getElementById('backBtn');
const shortcutsLink = document.getElementById('shortcutsLink');
const authSettingsLink = document.getElementById('authSettingsLink');

// Task search
const taskSearchDropdown = document.getElementById('taskSearchDropdown');
let cachedTasks = [];
let selectedSearchIndex = -1;
let selectedTask = null; // When user selects an existing task

// Supabase configuration (will be updated from content script or stored config)
const SUPABASE_CONFIG = {
  url: 'https://imwmxxihfyrnfuauxvmh.supabase.co',
  anonKey: 'eyJhbGciOiJIUzI1NiIsInR5cCI6IkpXVCJ9.eyJpc3MiOiJzdXBhYmFzZSIsInJlZiI6Imltd214eGloZnlybmZ1YXV4dm1oIiwicm9sZSI6ImFub24iLCJpYXQiOjE3NDQ4MjU4MzcsImV4cCI6MjA2MDQwMTgzN30.9QW7xvNmbCVj-2kRZRvQDMdG0Z_SNKLJyp3lzTAaEU0'
};

// NLP Task Parser (simplified version of the main app's parser)
const Priority = {
  High: 3,
  Medium: 2,
  Low: 1
};

function parseTaskInput(input) {
  if (!input || typeof input !== 'string') {
    return { title: '', priority: Priority.Medium };
  }

  let text = input.trim();
  const result = {
    title: '',
    priority: Priority.Medium,
    estimatedTime: undefined,
    dueDate: undefined,
    projectName: undefined,
    clientName: undefined
  };

  // Extract time estimates
  const timePatterns = [
    { pattern: /(\d+(?:\.\d+)?)\s*h(?:rs?|ours?)?\s*(\d+)\s*m(?:ins?)?/i, calc: (m) => m[1] * 3600 + m[2] * 60 },
    { pattern: /(\d+(?:\.\d+)?)\s*h(?:rs?|ours?)?/i, calc: (m) => m[1] * 3600 },
    { pattern: /(\d+)\s*m(?:ins?|inutes?)?/i, calc: (m) => m[1] * 60 },
  ];

  for (const { pattern, calc } of timePatterns) {
    const match = text.match(pattern);
    if (match) {
      result.estimatedTime = calc(match);
      text = text.replace(match[0], ' ');
      break;
    }
  }

  // Extract priority
  const priorityPatterns = [
    { pattern: /#(high|urgent|critical|important|!!)/i, priority: Priority.High },
    { pattern: /!!/g, priority: Priority.High },
    { pattern: /\(high\)/i, priority: Priority.High },
    { pattern: /#(medium|normal|mid)/i, priority: Priority.Medium },
    { pattern: /#(low|later|backlog)/i, priority: Priority.Low },
  ];

  for (const { pattern, priority } of priorityPatterns) {
    if (pattern.test(text)) {
      result.priority = priority;
      text = text.replace(pattern, ' ');
      break;
    }
  }

  // Extract due dates
  const today = new Date();
  const dueDatePatterns = [
    { pattern: /\b(today)\b/i, getDate: () => today.toISOString().split('T')[0] },
    { pattern: /\b(tomorrow)\b/i, getDate: () => {
      const d = new Date(today);
      d.setDate(d.getDate() + 1);
      return d.toISOString().split('T')[0];
    }},
    { pattern: /\b(next week)\b/i, getDate: () => {
      const d = new Date(today);
      d.setDate(d.getDate() + 7);
      return d.toISOString().split('T')[0];
    }},
    { pattern: /\b(monday|mon)\b/i, getDate: () => getNextDayOfWeek(1) },
    { pattern: /\b(tuesday|tue)\b/i, getDate: () => getNextDayOfWeek(2) },
    { pattern: /\b(wednesday|wed)\b/i, getDate: () => getNextDayOfWeek(3) },
    { pattern: /\b(thursday|thu)\b/i, getDate: () => getNextDayOfWeek(4) },
    { pattern: /\b(friday|fri)\b/i, getDate: () => getNextDayOfWeek(5) },
  ];

  for (const { pattern, getDate } of dueDatePatterns) {
    if (pattern.test(text)) {
      result.dueDate = getDate();
      text = text.replace(pattern, ' ');
      break;
    }
  }

  // Extract "for X" pattern for project/client
  const forMatch = text.match(/\s+for\s+["']?([A-Z][a-zA-Z]+(?:\s+[A-Z][a-zA-Z]+){0,2})["']?\s*$/i);
  if (forMatch) {
    result.projectName = forMatch[1].trim();
    text = text.replace(forMatch[0], ' ');
  }

  // Extract @mentions
  const clientMatch = text.match(/@"([^"]+)"|@(\S+)/);
  if (clientMatch) {
    result.clientName = clientMatch[1] || clientMatch[2];
    text = text.replace(clientMatch[0], ' ');
  }

  // Clean up title
  result.title = text
    .replace(/\s+/g, ' ')
    .replace(/^\s+|\s+$/g, '')
    .replace(/^[-–—]\s*/, '');

  if (result.title && result.title[0] === result.title[0].toLowerCase()) {
    result.title = result.title[0].toUpperCase() + result.title.slice(1);
  }

  return result;
}

function getNextDayOfWeek(dayOfWeek) {
  const today = new Date();
  const currentDay = today.getDay();
  let daysUntil = dayOfWeek - currentDay;
  if (daysUntil <= 0) daysUntil += 7;
  const nextDate = new Date(today);
  nextDate.setDate(today.getDate() + daysUntil);
  return nextDate.toISOString().split('T')[0];
}

function formatParsedTask(parsed) {
  const parts = [];
  
  if (parsed.priority === Priority.High) parts.push('🔴 High');
  else if (parsed.priority === Priority.Medium) parts.push('🟡 Medium');
  else if (parsed.priority === Priority.Low) parts.push('🟢 Low');
  
  if (parsed.estimatedTime) {
    const hours = Math.floor(parsed.estimatedTime / 3600);
    const mins = Math.floor((parsed.estimatedTime % 3600) / 60);
    if (hours > 0 && mins > 0) parts.push(`⏱ ${hours}h ${mins}m`);
    else if (hours > 0) parts.push(`⏱ ${hours}h`);
    else parts.push(`⏱ ${mins}m`);
  }
  
  if (parsed.dueDate) {
    const date = new Date(parsed.dueDate);
    parts.push(`📅 ${date.toLocaleDateString('en-US', { weekday: 'short', month: 'short', day: 'numeric' })}`);
  }
  
  if (parsed.clientName) parts.push(`👤 ${parsed.clientName}`);
  if (parsed.projectName) parts.push(`📁 ${parsed.projectName}`);
  
  return parts;
}

// DOM Elements
const taskInput = document.getElementById('taskInput');
const parsedPreview = document.getElementById('parsedPreview');
const createBtn = document.getElementById('createBtn');
const openAppBtn = document.getElementById('openAppBtn');
const mainForm = document.getElementById('mainForm');
const successMessage = document.getElementById('successMessage');

// Default app URL
const DEFAULT_APP_URL = 'https://app.focusops.net';

// Get current app URL from storage
async function getAppUrl() {
  const result = await chrome.storage.sync.get(['appUrl']);
  return result.appUrl || DEFAULT_APP_URL;
}

// ============ Settings Panel ============
function showSettings() {
  mainForm.classList.add('hidden');
  settingsPanel.classList.add('show');
  chrome.storage.sync.get(['appUrl'], (result) => {
    settingsAppUrl.value = result.appUrl || DEFAULT_APP_URL;
  });
}

function hideSettings() {
  settingsPanel.classList.remove('show');
  mainForm.classList.remove('hidden');
  taskInput.focus();
}

openSettingsBtn.addEventListener('click', showSettings);
backBtn.addEventListener('click', hideSettings);

settingsAppUrl.addEventListener('change', () => {
  chrome.storage.sync.set({ appUrl: settingsAppUrl.value });
});

shortcutsLink.addEventListener('click', (e) => {
  e.preventDefault();
  chrome.tabs.create({ url: 'chrome://extensions/shortcuts' });
});

authSettingsLink.addEventListener('click', (e) => {
  e.preventDefault();
  authRequired.classList.remove('show');
  settingsPanel.classList.add('show');
  chrome.storage.sync.get(['appUrl'], (result) => {
    settingsAppUrl.value = result.appUrl || DEFAULT_APP_URL;
  });
});

// ============ Direct Supabase API ============
// Fetch tasks directly from Supabase when app is not open

async function fetchTasksFromSupabase() {
  try {
    const authData = await chrome.storage.local.get(['authData']);
    if (!authData.authData?.access_token) {
      console.log('[FocusOps Popup] No auth token for Supabase');
      return null;
    }

    const token = authData.authData.access_token;
    
    // Check if token is expired
    if (authData.authData.expires_at) {
      const expiresAt = authData.authData.expires_at * 1000; // Convert to ms
      if (Date.now() >= expiresAt - 60000) { // Refresh if < 1 min left
        console.log('[FocusOps Popup] Token expired, trying refresh...');
        const refreshed = await refreshSupabaseToken();
        if (!refreshed) return null;
      }
    }

    // Fetch tasks from Supabase
    const response = await fetch(`${SUPABASE_CONFIG.url}/rest/v1/tasks?status=neq.completed&status=neq.archived&select=id,title,status,priority,project_id,due_date&limit=100`, {
      headers: {
        'apikey': SUPABASE_CONFIG.anonKey,
        'Authorization': `Bearer ${token}`,
        'Content-Type': 'application/json'
      }
    });

    if (!response.ok) {
      console.log('[FocusOps Popup] Supabase fetch failed:', response.status);
      return null;
    }

    const tasks = await response.json();
    
    // Also fetch projects for names
    const projectsResponse = await fetch(`${SUPABASE_CONFIG.url}/rest/v1/projects?select=id,name`, {
      headers: {
        'apikey': SUPABASE_CONFIG.anonKey,
        'Authorization': `Bearer ${token}`,
        'Content-Type': 'application/json'
      }
    });

    let projects = {};
    if (projectsResponse.ok) {
      const projectsList = await projectsResponse.json();
      projectsList.forEach(p => { projects[p.id] = p; });
    }

    // Map to expected format
    const mappedTasks = tasks.map(t => ({
      id: t.id,
      title: t.title,
      status: t.status,
      priority: t.priority,
      projectId: t.project_id,
      projectName: t.project_id && projects[t.project_id] ? projects[t.project_id].name : null,
      dueDate: t.due_date
    }));

    console.log('[FocusOps Popup] Fetched', mappedTasks.length, 'tasks from Supabase');
    return mappedTasks;

  } catch (e) {
    console.error('[FocusOps Popup] Supabase fetch error:', e);
    return null;
  }
}

async function refreshSupabaseToken() {
  try {
    const authData = await chrome.storage.local.get(['authData']);
    if (!authData.authData?.refresh_token) {
      return false;
    }

    const response = await fetch(`${SUPABASE_CONFIG.url}/auth/v1/token?grant_type=refresh_token`, {
      method: 'POST',
      headers: {
        'apikey': SUPABASE_CONFIG.anonKey,
        'Content-Type': 'application/json'
      },
      body: JSON.stringify({
        refresh_token: authData.authData.refresh_token
      })
    });

    if (!response.ok) {
      console.log('[FocusOps Popup] Token refresh failed');
      return false;
    }

    const data = await response.json();
    
    // Update stored auth data
    await chrome.storage.local.set({
      authData: {
        access_token: data.access_token,
        refresh_token: data.refresh_token,
        expires_at: data.expires_at,
        user: data.user
      },
      authStatus: {
        authenticated: true,
        userId: data.user?.id,
        timestamp: Date.now()
      }
    });

    console.log('[FocusOps Popup] Token refreshed successfully');
    return true;

  } catch (e) {
    console.error('[FocusOps Popup] Token refresh error:', e);
    return false;
  }
}

async function createTaskInSupabase(taskData) {
  try {
    const authData = await chrome.storage.local.get(['authData']);
    if (!authData.authData?.access_token) {
      console.log('[FocusOps Popup] No auth token for task creation');
      return null;
    }

    const response = await fetch(`${SUPABASE_CONFIG.url}/rest/v1/tasks`, {
      method: 'POST',
      headers: {
        'apikey': SUPABASE_CONFIG.anonKey,
        'Authorization': `Bearer ${authData.authData.access_token}`,
        'Content-Type': 'application/json',
        'Prefer': 'return=representation'
      },
      body: JSON.stringify({
        title: taskData.title,
        status: 'todo',
        priority: taskData.priority || 2,
        estimated_time: taskData.estimatedTime || null,
        due_date: taskData.dueDate || null,
        user_id: authData.authData.user?.id
      })
    });

    if (!response.ok) {
      console.log('[FocusOps Popup] Task creation failed:', response.status);
      return null;
    }

    const [task] = await response.json();
    console.log('[FocusOps Popup] Created task in Supabase:', task.id);
    return task;

  } catch (e) {
    console.error('[FocusOps Popup] Task creation error:', e);
    return null;
  }
}

// ============ Task Search ============
async function fetchTasks() {
  const appUrl = await getAppUrl();
  console.log('[FocusOps Popup] Fetching tasks from:', appUrl);
  
  try {
    // First try to get tasks from an open FocusOps tab
    const tabs = await chrome.tabs.query({ url: `${appUrl}/*` });
    console.log('[FocusOps Popup] Found', tabs.length, 'FocusOps tabs');
    
    if (tabs.length > 0) {
      try {
        const response = await Promise.race([
          chrome.tabs.sendMessage(tabs[0].id, { type: 'FOCUSOPS_GET_TASKS' }),
          new Promise((_, reject) => setTimeout(() => reject(new Error('Timeout')), 3000))
        ]);
        
        if (response && response.tasks && response.tasks.length > 0) {
          cachedTasks = response.tasks;
          console.log('[FocusOps Popup] Cached', cachedTasks.length, 'tasks from tab');
          // Save to chrome storage for offline access
          chrome.storage.local.set({ cachedTasks: cachedTasks, cachedTasksTimestamp: Date.now() });
          return response.tasks;
        }
      } catch (e) {
        console.log('[FocusOps Popup] Could not get tasks from tab:', e.message);
      }
    }
    
    // If no tabs open or no tasks from tab, try direct Supabase fetch
    console.log('[FocusOps Popup] Trying direct Supabase fetch...');
    const supabaseTasks = await fetchTasksFromSupabase();
    if (supabaseTasks && supabaseTasks.length > 0) {
      cachedTasks = supabaseTasks;
      chrome.storage.local.set({ cachedTasks: cachedTasks, cachedTasksTimestamp: Date.now() });
      return supabaseTasks;
    }
    
    // Fall back to cached tasks from storage
    try {
      const stored = await chrome.storage.local.get(['cachedTasks', 'cachedTasksTimestamp']);
      if (stored.cachedTasks && stored.cachedTasks.length > 0) {
        const age = Date.now() - (stored.cachedTasksTimestamp || 0);
        const ONE_DAY = 24 * 60 * 60 * 1000;
        
        // Use cache if less than 1 day old
        if (age < ONE_DAY) {
          cachedTasks = stored.cachedTasks;
          console.log('[FocusOps Popup] Using cached tasks from storage:', cachedTasks.length, '(age:', Math.round(age / 60000), 'min)');
          return cachedTasks;
        } else {
          console.log('[FocusOps Popup] Cache too old, skipping');
        }
      }
    } catch (e) {
      console.log('[FocusOps Popup] No cached tasks in storage');
    }
    
  } catch (e) {
    console.log('[FocusOps Popup] Fetch tasks error:', e.message);
  }
  
  return cachedTasks;
}

function searchTasks(query) {
  if (!query || query.length < 2) return [];
  const lower = query.toLowerCase();
  const results = cachedTasks
    .filter(t => t.title.toLowerCase().includes(lower) || (t.projectName && t.projectName.toLowerCase().includes(lower)))
    .slice(0, 5);
  console.log('[FocusOps Popup] Search for "' + query + '" found', results.length, 'results from', cachedTasks.length, 'cached tasks');
  return results;
}

function renderTaskDropdown(tasks, query) {
  if (tasks.length === 0 && query.length >= 2) {
    taskSearchDropdown.innerHTML = `
      <div class="task-search-item task-search-new" data-action="create">
        <div class="task-search-title">➕ Create new task: "${query}"</div>
      </div>
    `;
    taskSearchDropdown.classList.add('show');
  } else if (tasks.length > 0) {
    let html = tasks.map((t, i) => `
      <div class="task-search-item${i === selectedSearchIndex ? ' selected' : ''}" data-task-id="${t.id}">
        <div class="task-search-title">${escapeHtml(t.title)}</div>
        <div class="task-search-meta">
          ${t.projectName ? `<span>📁 ${escapeHtml(t.projectName)}</span>` : ''}
          ${t.status ? `<span>${t.status}</span>` : ''}
        </div>
      </div>
    `).join('');
    html += `
      <div class="task-search-item task-search-new" data-action="create">
        <div class="task-search-title">➕ Create new task</div>
      </div>
    `;
    taskSearchDropdown.innerHTML = html;
    taskSearchDropdown.classList.add('show');
  } else {
    taskSearchDropdown.classList.remove('show');
  }
}

function escapeHtml(text) {
  const div = document.createElement('div');
  div.textContent = text;
  return div.innerHTML;
}

function hideTaskDropdown() {
  taskSearchDropdown.classList.remove('show');
  selectedSearchIndex = -1;
}

// Task dropdown click handler
taskSearchDropdown.addEventListener('click', (e) => {
  const item = e.target.closest('.task-search-item');
  if (!item) return;
  
  if (item.dataset.action === 'create') {
    selectedTask = null;
    hideTaskDropdown();
    return;
  }
  
  const taskId = item.dataset.taskId;
  const task = cachedTasks.find(t => t.id === taskId);
  if (task) {
    selectedTask = task;
    taskInput.value = task.title;
    hideTaskDropdown();
    updatePreviewForExistingTask(task);
  }
});

function updatePreviewForExistingTask(task) {
  let html = `<strong>📌 Existing Task: "${escapeHtml(task.title)}"</strong><br>`;
  const parts = [];
  if (task.priority === 3) parts.push('<span class="parsed-item priority-high">🔴 High</span>');
  else if (task.priority === 2) parts.push('<span class="parsed-item priority-medium">🟡 Medium</span>');
  else if (task.priority === 1) parts.push('<span class="parsed-item priority-low">🟢 Low</span>');
  if (task.projectName) parts.push(`<span class="parsed-item">📁 ${escapeHtml(task.projectName)}</span>`);
  if (task.dueDate) parts.push(`<span class="parsed-item">📅 ${new Date(task.dueDate).toLocaleDateString()}</span>`);
  html += parts.join('');
  parsedPreview.innerHTML = html;
  parsedPreview.classList.add('has-data');
}

// ============ Authentication ============
async function checkAuthStatus() {
  const appUrl = await getAppUrl();
  
  // Show checking state
  authChecking.classList.add('show');
  authRequired.classList.remove('show');
  mainForm.classList.add('hidden');
  settingsPanel.classList.remove('show');
  
  // Set a maximum timeout for the entire auth check
  const authTimeout = setTimeout(() => {
    console.log('Auth check timed out, checking cache...');
    checkCacheAndProceed(appUrl);
  }, 1500);
  
  try {
    // First check cached auth status (valid for 7 days)
    const cached = await chrome.storage.local.get(['authStatus', 'authData']);
    if (cached.authStatus && cached.authStatus.authenticated) {
      const age = Date.now() - cached.authStatus.timestamp;
      const SEVEN_DAYS = 7 * 24 * 60 * 60 * 1000;
      
      if (age < SEVEN_DAYS) {
        clearTimeout(authTimeout);
        showMainForm();
        refreshAuthInBackground(appUrl);
        fetchTasks(); // Pre-fetch tasks
        return;
      }
    }
    
    // Try to communicate with the app
    const tabs = await chrome.tabs.query({ url: `${appUrl}/*` });
    
    if (tabs.length > 0) {
      try {
        const response = await Promise.race([
          chrome.tabs.sendMessage(tabs[0].id, { type: 'FOCUSOPS_CHECK_AUTH' }),
          new Promise((_, reject) => setTimeout(() => reject(new Error('Timeout')), 1000))
        ]);
        
        clearTimeout(authTimeout);
        
        if (response && response.authenticated) {
          showMainForm();
          
          // Store auth data for offline Supabase access
          const authStorage = {
            authStatus: {
              authenticated: true,
              userId: response.userId,
              timestamp: Date.now()
            }
          };
          
          // Store Supabase tokens if provided
          if (response.authData) {
            authStorage.authData = response.authData;
          }
          
          await chrome.storage.local.set(authStorage);
          fetchTasks();
          return;
        }
      } catch (e) {
        console.log('Could not communicate with app tab:', e.message);
      }
    }
    
    // No app tab open - check if we have valid stored auth for direct Supabase access
    if (cached.authData?.access_token) {
      // Check if token might still be valid
      const expiresAt = cached.authData.expires_at ? cached.authData.expires_at * 1000 : 0;
      
      if (Date.now() < expiresAt - 60000) {
        // Token not expired yet
        clearTimeout(authTimeout);
        console.log('[FocusOps Popup] Using stored auth token');
        showMainForm();
        fetchTasks();
        return;
      } else if (cached.authData.refresh_token) {
        // Try to refresh the token
        console.log('[FocusOps Popup] Trying to refresh token...');
        const refreshed = await refreshSupabaseToken();
        if (refreshed) {
          clearTimeout(authTimeout);
          showMainForm();
          fetchTasks();
          return;
        }
      }
    }
    
    clearTimeout(authTimeout);
    showAuthRequired();
    
  } catch (error) {
    clearTimeout(authTimeout);
    console.error('Auth check error:', error);
    checkCacheAndProceed(appUrl);
  }
}

async function checkCacheAndProceed(appUrl) {
  const cached = await chrome.storage.local.get(['authStatus']);
  if (cached.authStatus && cached.authStatus.authenticated) {
    showMainForm();
    fetchTasks();
  } else {
    showAuthRequired();
  }
}

async function refreshAuthInBackground(appUrl) {
  try {
    const tabs = await chrome.tabs.query({ url: `${appUrl}/*` });
    if (tabs.length > 0) {
      const response = await Promise.race([
        chrome.tabs.sendMessage(tabs[0].id, { type: 'FOCUSOPS_CHECK_AUTH' }),
        new Promise((_, reject) => setTimeout(() => reject(new Error('Timeout')), 2000))
      ]);
      
      if (response && response.authenticated) {
        const authStorage = {
          authStatus: {
            authenticated: true,
            userId: response.userId,
            timestamp: Date.now()
          }
        };
        
        // Store Supabase tokens if provided
        if (response.authData) {
          authStorage.authData = response.authData;
        }
        
        await chrome.storage.local.set(authStorage);
      }
    }
  } catch (e) {
    // Silent fail
  }
}

function showMainForm() {
  authChecking.classList.remove('show');
  authRequired.classList.remove('show');
  mainForm.classList.remove('hidden');
  settingsPanel.classList.remove('show');
  taskInput.focus();
}

function showAuthRequired() {
  authChecking.classList.remove('show');
  authRequired.classList.add('show');
  mainForm.classList.add('hidden');
  settingsPanel.classList.remove('show');
}

// Login button handler
loginBtn.addEventListener('click', async () => {
  const appUrl = await getAppUrl();
  chrome.tabs.create({ url: appUrl });
  window.close();
});

// Skip auth button handler
const skipAuthBtn = document.getElementById('skipAuthBtn');
skipAuthBtn.addEventListener('click', async () => {
  await chrome.storage.local.set({
    authStatus: {
      authenticated: true,
      userId: 'user-skipped-auth',
      timestamp: Date.now(),
      skipped: true
    }
  });
  showMainForm();
  fetchTasks();
});

// ============ Initialize ============
// Load saved settings and check auth
chrome.storage.sync.get(['appUrl'], (result) => {
  settingsAppUrl.value = result.appUrl || DEFAULT_APP_URL;
  
  // Check for selected text passed from content script
  chrome.storage.local.get(['selectedText'], (localResult) => {
    if (localResult.selectedText) {
      taskInput.value = localResult.selectedText;
      chrome.storage.local.remove('selectedText');
      updatePreview();
    }
  });
  
  // Check authentication
  checkAuthStatus();
});

// Update preview as user types with search
let searchDebounce = null;
taskInput.addEventListener('input', () => {
  selectedTask = null; // Clear selected existing task when typing
  updatePreview();
  
  // Debounced search
  clearTimeout(searchDebounce);
  searchDebounce = setTimeout(async () => {
    const query = taskInput.value.trim();
    
    // If cache is empty, try to fetch tasks first
    if (cachedTasks.length === 0 && query.length >= 2) {
      await fetchTasks();
    }
    
    const results = searchTasks(query);
    renderTaskDropdown(results, query);
  }, 150);
});

// Handle keyboard navigation in dropdown
taskInput.addEventListener('keydown', (e) => {
  const items = taskSearchDropdown.querySelectorAll('.task-search-item');
  
  if (e.key === 'ArrowDown' && taskSearchDropdown.classList.contains('show')) {
    e.preventDefault();
    selectedSearchIndex = Math.min(selectedSearchIndex + 1, items.length - 1);
    items.forEach((item, i) => item.classList.toggle('selected', i === selectedSearchIndex));
  } else if (e.key === 'ArrowUp' && taskSearchDropdown.classList.contains('show')) {
    e.preventDefault();
    selectedSearchIndex = Math.max(selectedSearchIndex - 1, 0);
    items.forEach((item, i) => item.classList.toggle('selected', i === selectedSearchIndex));
  } else if (e.key === 'Enter') {
    if (taskSearchDropdown.classList.contains('show') && selectedSearchIndex >= 0) {
      e.preventDefault();
      items[selectedSearchIndex].click();
    } else {
      createTask();
    }
  } else if (e.key === 'Escape') {
    hideTaskDropdown();
  }
});

// Hide dropdown when clicking outside
document.addEventListener('click', (e) => {
  if (!taskInput.contains(e.target) && !taskSearchDropdown.contains(e.target)) {
    hideTaskDropdown();
  }
});

function updatePreview() {
  const text = taskInput.value;
  if (!text.trim()) {
    parsedPreview.innerHTML = 'Type a task to see parsed properties...';
    parsedPreview.classList.remove('has-data');
    return;
  }
  
  const parsed = parseTaskInput(text);
  const parts = formatParsedTask(parsed);
  
  if (parts.length === 0 && !parsed.title) {
    parsedPreview.innerHTML = 'Type a task to see parsed properties...';
    parsedPreview.classList.remove('has-data');
    return;
  }
  
  let html = `<strong>"${escapeHtml(parsed.title)}"</strong><br>`;
  html += parts.map(part => {
    let className = 'parsed-item';
    if (part.includes('High')) className += ' priority-high';
    else if (part.includes('Medium')) className += ' priority-medium';
    else if (part.includes('Low')) className += ' priority-low';
    return `<span class="${className}">${part}</span>`;
  }).join('');
  
  parsedPreview.innerHTML = html;
  parsedPreview.classList.add('has-data');
}

// Create task
createBtn.addEventListener('click', createTask);

async function createTask() {
  const text = taskInput.value.trim();
  if (!text) return;
  
  const appUrl = await getAppUrl();
  
  // If an existing task was selected, start tracking it instead of creating new
  if (selectedTask) {
    // Focus existing tab and start tracking this task
    const tabs = await chrome.tabs.query({ url: `${appUrl}/*` });
    if (tabs.length > 0) {
      chrome.tabs.update(tabs[0].id, { active: true });
      chrome.tabs.sendMessage(tabs[0].id, {
        type: 'FOCUSOPS_START_TASK',
        taskId: selectedTask.id
      });
    } else {
      chrome.tabs.create({ url: `${appUrl}/?taskId=${selectedTask.id}` });
    }
    
    mainForm.classList.add('hidden');
    successMessage.querySelector('.success-text').textContent = 'Task Selected!';
    successMessage.classList.add('show');
    setTimeout(() => window.close(), 1000);
    return;
  }
  
  const parsed = parseTaskInput(text);
  
  // Check if FocusOps tab is open
  const tabs = await chrome.tabs.query({ url: `${appUrl}/*` });
  
  if (tabs.length > 0) {
    // Tab is open - use messaging
    chrome.tabs.update(tabs[0].id, { active: true });
    chrome.tabs.sendMessage(tabs[0].id, {
      type: 'FOCUSOPS_QUICK_ADD',
      taskText: text,
      parsed: parsed
    });
    
    mainForm.classList.add('hidden');
    successMessage.querySelector('.success-text').textContent = 'Task Created!';
    successMessage.classList.add('show');
    setTimeout(() => window.close(), 1000);
  } else {
    // No tab open - try direct Supabase create
    const createdTask = await createTaskInSupabase({
      title: parsed.title || text,
      priority: parsed.priority,
      estimatedTime: parsed.estimatedTime,
      dueDate: parsed.dueDate
    });
    
    if (createdTask) {
      // Success - task created directly
      mainForm.classList.add('hidden');
      successMessage.querySelector('.success-text').textContent = 'Task Created!';
      successMessage.classList.add('show');
      
      // Update cached tasks
      cachedTasks.unshift({
        id: createdTask.id,
        title: createdTask.title,
        status: createdTask.status,
        priority: createdTask.priority,
        projectId: null,
        projectName: null,
        dueDate: createdTask.due_date
      });
      chrome.storage.local.set({ cachedTasks: cachedTasks, cachedTasksTimestamp: Date.now() });
      
      setTimeout(() => window.close(), 1000);
    } else {
      // Fallback - open app with prefilled task
      const params = new URLSearchParams({
        'shortcut': 'quick-add',
        'prefill': text
      });
      chrome.tabs.create({ url: `${appUrl}/?${params.toString()}` });
      
      mainForm.classList.add('hidden');
      successMessage.querySelector('.success-text').textContent = 'Opening app...';
      successMessage.classList.add('show');
      setTimeout(() => window.close(), 1000);
    }
  }
}

// Open app button
openAppBtn.addEventListener('click', async () => {
  const appUrl = await getAppUrl();
  chrome.tabs.create({ url: appUrl });
  window.close();
});

// Focus input on load
taskInput.focus();

// Time Tracking Elements
const trackingSection = document.getElementById('trackingSection');
const trackingStatus = document.getElementById('trackingStatus');
const trackingTimer = document.getElementById('trackingTimer');
const trackingTask = document.getElementById('trackingTask');
const startTrackingBtn = document.getElementById('startTrackingBtn');
const stopTrackingBtn = document.getElementById('stopTrackingBtn');

let timerInterval = null;

// Load current tracking state from storage
function loadTrackingState() {
  chrome.storage.local.get(['activeTimeLog', 'trackingTaskTitle'], (result) => {
    if (result.activeTimeLog && result.activeTimeLog.startTime) {
      // Timer is running
      updateTrackingUI(true, result.trackingTaskTitle || 'Unknown task', result.activeTimeLog.startTime);
      startTimerDisplay(result.activeTimeLog.startTime);
    } else {
      updateTrackingUI(false);
    }
  });
}

function updateTrackingUI(isTracking, taskTitle = '', startTime = null) {
  if (isTracking) {
    trackingStatus.textContent = 'Tracking...';
    trackingStatus.classList.add('active');
    trackingTimer.classList.add('active');
    trackingTask.textContent = taskTitle;
    startTrackingBtn.style.display = 'none';
    stopTrackingBtn.style.display = 'block';
  } else {
    trackingStatus.textContent = 'Not tracking';
    trackingStatus.classList.remove('active');
    trackingTimer.classList.remove('active');
    trackingTimer.textContent = '00:00:00';
    trackingTask.textContent = '';
    startTrackingBtn.style.display = 'block';
    stopTrackingBtn.style.display = 'none';
  }
}

function startTimerDisplay(startTime) {
  if (timerInterval) clearInterval(timerInterval);
  
  function updateTimer() {
    const elapsed = Math.floor((Date.now() - new Date(startTime).getTime()) / 1000);
    const hours = Math.floor(elapsed / 3600);
    const mins = Math.floor((elapsed % 3600) / 60);
    const secs = elapsed % 60;
    trackingTimer.textContent = 
      `${hours.toString().padStart(2, '0')}:${mins.toString().padStart(2, '0')}:${secs.toString().padStart(2, '0')}`;
  }
  
  updateTimer();
  timerInterval = setInterval(updateTimer, 1000);
}

// Start tracking - needs to communicate with the app
startTrackingBtn.addEventListener('click', async () => {
  const text = taskInput.value.trim();
  
  // If an existing task was selected, use that
  const taskToTrack = selectedTask || (text ? { title: parseTaskInput(text).title || text } : null);
  
  if (!taskToTrack) {
    taskInput.placeholder = 'Enter a task to track...';
    taskInput.focus();
    return;
  }
  
  const parsed = selectedTask ? {} : parseTaskInput(text);
  const appUrl = await getAppUrl();
  const startTime = new Date().toISOString();
  
  // Store tracking state
  await chrome.storage.local.set({
    activeTimeLog: {
      startTime: startTime,
      taskText: selectedTask ? selectedTask.title : text,
      taskId: selectedTask ? selectedTask.id : null,
      parsed: parsed
    },
    trackingTaskTitle: taskToTrack.title
  });
  
  // Update UI
  updateTrackingUI(true, taskToTrack.title, startTime);
  startTimerDisplay(startTime);
  
  // Try to notify the app
  const tabs = await chrome.tabs.query({ url: `${appUrl}/*` });
  if (tabs.length > 0) {
    chrome.tabs.sendMessage(tabs[0].id, {
      type: 'FOCUSOPS_START_TRACKING',
      taskText: selectedTask ? selectedTask.title : text,
      taskId: selectedTask ? selectedTask.id : null,
      parsed: parsed,
      startTime: startTime
    });
  }
});

// Stop tracking
stopTrackingBtn.addEventListener('click', async () => {
  const result = await chrome.storage.local.get(['activeTimeLog', 'trackingTaskTitle']);
  const appUrl = await getAppUrl();
  
  if (result.activeTimeLog) {
    const endTime = new Date().toISOString();
    const timeLog = {
      ...result.activeTimeLog,
      endTime: endTime,
      duration: Math.floor((new Date(endTime).getTime() - new Date(result.activeTimeLog.startTime).getTime()) / 1000)
    };
    
    // Try to send to app to save the time log
    const tabs = await chrome.tabs.query({ url: `${appUrl}/*` });
    if (tabs.length > 0) {
      chrome.tabs.sendMessage(tabs[0].id, {
        type: 'FOCUSOPS_STOP_TRACKING',
        timeLog: timeLog
      });
    } else {
      // Store pending time log to be synced when app opens
      chrome.storage.local.get(['pendingTimeLogs'], (r) => {
        const pending = r.pendingTimeLogs || [];
        pending.push(timeLog);
        chrome.storage.local.set({ pendingTimeLogs: pending });
      });
    }
  }
  
  // Clear tracking state
  await chrome.storage.local.remove(['activeTimeLog', 'trackingTaskTitle']);
  if (timerInterval) clearInterval(timerInterval);
  updateTrackingUI(false);
});

// Load tracking state on popup open
loadTrackingState();
