// FocusOps Quick Add - Content Script
// Runs on all pages to enable communication with the app

// Listen for messages from the background script or popup
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  // Check authentication status
  if (message.type === 'FOCUSOPS_CHECK_AUTH') {
    // Only check on FocusOps pages
    const hostname = window.location.hostname;
    const isFocusOpsPage = hostname.includes('focusops') || hostname === 'localhost';
    
    if (isFocusOpsPage) {
      // Check for auth data in localStorage (Supabase stores session there)
      try {
        let isAuthenticated = false;
        let userId = null;
        let authData = null;
        
        // Check the FocusOps-specific auth token (configured in supabaseClient.ts)
        const focusOpsAuthToken = localStorage.getItem('focusops-auth-token');
        if (focusOpsAuthToken) {
          try {
            const data = JSON.parse(focusOpsAuthToken);
            if (data?.access_token || data?.user?.id) {
              isAuthenticated = true;
              userId = data?.user?.id || data?.sub;
              authData = {
                access_token: data.access_token,
                refresh_token: data.refresh_token,
                expires_at: data.expires_at,
                user: data.user
              };
            }
          } catch (e) {
            // Not JSON or malformed
          }
        }
        
        // Also check for standard Supabase auth keys (sb-*-auth-token pattern)
        if (!isAuthenticated) {
          const authKeys = Object.keys(localStorage).filter(key => 
            key.startsWith('sb-') && key.includes('-auth-token')
          );
          
          for (const key of authKeys) {
            try {
              const data = JSON.parse(localStorage.getItem(key));
              if (data?.access_token || data?.user?.id) {
                isAuthenticated = true;
                userId = data?.user?.id || data?.sub;
                authData = {
                  access_token: data.access_token,
                  refresh_token: data.refresh_token,
                  expires_at: data.expires_at,
                  user: data.user
                };
                break;
              }
            } catch (e) {
              // Not JSON, skip
            }
          }
        }
        
        // Check for the last known user ID (set by AuthContext)
        if (!isAuthenticated) {
          const lastUserId = localStorage.getItem('focusops_last_user_id');
          if (lastUserId) {
            // User was previously authenticated - check if token still exists
            isAuthenticated = !!focusOpsAuthToken;
            userId = lastUserId;
          }
        }
        
        console.log('[FocusOps Content] Auth check result:', { isAuthenticated, userId });
        
        sendResponse({
          authenticated: isAuthenticated,
          userId: userId,
          authData: authData // Send auth tokens to extension for offline use
        });
        
      } catch (e) {
        console.error('[FocusOps Content] Auth check error:', e);
        sendResponse({ authenticated: false });
      }
    } else {
      sendResponse({ authenticated: false, reason: 'not_on_app' });
    }
    return false; // Synchronous response
  }
  
  // Get Supabase config from the app
  if (message.type === 'FOCUSOPS_GET_SUPABASE_CONFIG') {
    const hostname = window.location.hostname;
    const isFocusOpsPage = hostname.includes('focusops') || hostname === 'localhost';
    
    if (isFocusOpsPage) {
      // Try to extract Supabase URL from meta tags or global config
      // The app exposes these via import.meta.env, but we can infer from any Supabase calls
      // For now, we'll use the known production values
      sendResponse({
        supabaseUrl: 'https://imwmxxihfyrnfuauxvmh.supabase.co',
        // Note: anon key is safe to expose - it's public
        supabaseAnonKey: 'eyJhbGciOiJIUzI1NiIsInR5cCI6IkpXVCJ9.eyJpc3MiOiJzdXBhYmFzZSIsInJlZiI6Imltd214eGloZnlybmZ1YXV4dm1oIiwicm9sZSI6ImFub24iLCJpYXQiOjE3NDQ4MjU4MzcsImV4cCI6MjA2MDQwMTgzN30.9QW7xvNmbCVj-2kRZRvQDMdG0Z_SNKLJyp3lzTAaEU0'
      });
    } else {
      sendResponse({ error: 'not_on_app' });
    }
    return false;
  }
  
  if (message.type === 'FOCUSOPS_QUICK_ADD') {
    // If we're on the FocusOps app, trigger the command palette with the task
    const hostname = window.location.hostname;
    const isFocusOpsPage = hostname.includes('focusops') || hostname === 'localhost';
    
    if (isFocusOpsPage) {
      // Dispatch custom event that the app can listen for
      window.dispatchEvent(new CustomEvent('focusops-quick-add', {
        detail: {
          taskText: message.taskText,
          parsed: message.parsed
        }
      }));
      
      // Also try to store in sessionStorage for the app to pick up
      try {
        sessionStorage.setItem('shared-task-content', message.taskText);
        // Trigger command palette open via URL param
        const url = new URL(window.location.href);
        url.searchParams.set('shortcut', 'quick-add');
        window.history.pushState({}, '', url.toString());
        // Dispatch popstate to trigger React's URL param handling
        window.dispatchEvent(new PopStateEvent('popstate'));
      } catch (e) {
        console.error('[FocusOps Content] Error setting sessionStorage:', e);
      }
    }
    
    sendResponse({ success: true });
  }
  
  // Handle time tracking start from extension
  if (message.type === 'FOCUSOPS_START_TRACKING') {
    window.dispatchEvent(new CustomEvent('focusops-start-tracking', {
      detail: {
        taskText: message.taskText,
        parsed: message.parsed,
        startTime: message.startTime
      }
    }));
    sendResponse({ success: true });
  }
  
  // Handle time tracking stop from extension
  if (message.type === 'FOCUSOPS_STOP_TRACKING') {
    window.dispatchEvent(new CustomEvent('focusops-stop-tracking', {
      detail: {
        timeLog: message.timeLog
      }
    }));
    sendResponse({ success: true });
  }
  
  // Handle task list request from extension
  if (message.type === 'FOCUSOPS_GET_TASKS') {
    const hostname = window.location.hostname;
    const isFocusOpsPage = hostname.includes('focusops') || hostname === 'localhost';
    
    if (isFocusOpsPage) {
      // Use async handler for IndexedDB access
      (async () => {
        try {
          let tasks = [];
          let projects = {};
          
          // Try IndexedDB first (primary storage)
          try {
            const dbRequest = indexedDB.open('focusops-database');
            const db = await new Promise((resolve, reject) => {
              dbRequest.onsuccess = () => resolve(dbRequest.result);
              dbRequest.onerror = () => reject(dbRequest.error);
            });
            
            if (db.objectStoreNames.contains('tasks')) {
              const tx = db.transaction(['tasks', 'projects'], 'readonly');
              const taskStore = tx.objectStore('tasks');
              const projectStore = tx.objectStore('projects');
              
              // Get all projects first
              const projectsList = await new Promise((resolve) => {
                const req = projectStore.getAll();
                req.onsuccess = () => resolve(req.result || []);
                req.onerror = () => resolve([]);
              });
              projectsList.forEach(p => { projects[p.id] = p; });
              
              // Get all tasks
              const allTasks = await new Promise((resolve) => {
                const req = taskStore.getAll();
                req.onsuccess = () => resolve(req.result || []);
                req.onerror = () => resolve([]);
              });
              
              tasks = allTasks
                .filter(t => t.status !== 'completed' && t.status !== 'archived')
                .map(t => ({
                  id: t.id,
                  title: t.title,
                  status: t.status,
                  priority: t.priority,
                  projectId: t.projectId,
                  projectName: t.projectId && projects[t.projectId] ? projects[t.projectId].name : null,
                  dueDate: t.dueDate
                }))
                .slice(0, 100);
              
              db.close();
              
              if (tasks.length > 0) {
                console.log('[FocusOps Content] Found', tasks.length, 'tasks in IndexedDB');
                sendResponse({ tasks });
                return;
              }
            }
          } catch (idbError) {
            console.log('[FocusOps Content] IndexedDB error, falling back to localStorage:', idbError.message);
          }
          
          // Fallback to localStorage (legacy storage)
          // Try multiple possible key names
          const taskKeys = ['tasks', 'focusops-tasks', 'focusops_tasks'];
          const projectKeys = ['projects', 'focusops-projects', 'focusops_projects'];
          
          let tasksData = null;
          let projectsData = null;
          
          for (const key of projectKeys) {
            projectsData = localStorage.getItem(key);
            if (projectsData) break;
          }
          
          for (const key of taskKeys) {
            tasksData = localStorage.getItem(key);
            if (tasksData) break;
          }
          
          if (projectsData) {
            try {
              const projectsList = JSON.parse(projectsData);
              projectsList.forEach(p => { projects[p.id] = p; });
            } catch (e) {}
          }
          
          if (tasksData) {
            const allTasks = JSON.parse(tasksData);
            tasks = allTasks
              .filter(t => t.status !== 'completed' && t.status !== 'archived')
              .map(t => ({
                id: t.id,
                title: t.title,
                status: t.status,
                priority: t.priority,
                projectId: t.projectId,
                projectName: t.projectId && projects[t.projectId] ? projects[t.projectId].name : null,
                dueDate: t.dueDate
              }))
              .slice(0, 100);
          }
          
          console.log('[FocusOps Content] Found', tasks.length, 'tasks in localStorage');
          sendResponse({ tasks });
        } catch (e) {
          console.error('[FocusOps Content] Error getting tasks:', e);
          sendResponse({ tasks: [] });
        }
      })();
      return true; // Keep message channel open for async response
    } else {
      sendResponse({ tasks: [] });
    }
    return false;
  }
  
  // Handle start task request (to open a specific task in focus mode)
  if (message.type === 'FOCUSOPS_START_TASK') {
    window.dispatchEvent(new CustomEvent('focusops-start-task', {
      detail: {
        taskId: message.taskId
      }
    }));
    sendResponse({ success: true });
  }
  
  return true;
});

// Add keyboard shortcut hint to the page if on FocusOps
const hostname = window.location.hostname;
const isFocusOpsPage = hostname.includes('focusops') || hostname === 'localhost';

if (isFocusOpsPage) {
  // Listen for the custom event from extension
  window.addEventListener('focusops-quick-add', (event) => {
    const detail = event.detail;
    console.log('[FocusOps Content] Received quick add event:', detail);
    
    // The app's useEffect should pick up the sessionStorage value
    // But we can also try to programmatically open the command palette
    const cmdK = new KeyboardEvent('keydown', {
      key: 'k',
      code: 'KeyK',
      ctrlKey: true,
      bubbles: true
    });
    document.dispatchEvent(cmdK);
  });
}

// Double-click to add selected text as task (optional feature)
let lastDoubleClick = 0;
document.addEventListener('dblclick', (e) => {
  // Require Ctrl/Cmd + double-click
  if (!e.ctrlKey && !e.metaKey) return;
  
  const selection = window.getSelection()?.toString().trim();
  if (selection && selection.length > 2 && selection.length < 200) {
    // Debounce
    const now = Date.now();
    if (now - lastDoubleClick < 500) return;
    lastDoubleClick = now;
    
    // Send to background to open popup with selection
    chrome.runtime.sendMessage({
      type: 'QUICK_ADD_FROM_SELECTION',
      text: selection
    });
  }
});

console.log('[FocusOps Content] Content script loaded');
