// FocusOps Quick Add - Background Service Worker

// Handle keyboard shortcut commands
chrome.commands.onCommand.addListener(async (command) => {
  console.log('[FocusOps] Command received:', command);
  
  if (command === 'quick-add-selected') {
    // Get selected text from active tab
    const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
    
    if (tab?.id) {
      try {
        const results = await chrome.scripting.executeScript({
          target: { tabId: tab.id },
          function: () => window.getSelection()?.toString() || ''
        });
        
        const selectedText = results[0]?.result || '';
        
        if (selectedText) {
          // Store selected text for popup to use
          await chrome.storage.local.set({ selectedText });
        }
        
        // Open popup (this happens automatically for _execute_action, but we need it for custom commands)
        chrome.action.openPopup();
      } catch (error) {
        console.error('[FocusOps] Error getting selection:', error);
        // Still open popup even if we couldn't get selection
        chrome.action.openPopup();
      }
    }
  }
});

// Handle messages from content scripts
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  console.log('[FocusOps] Message received:', message);
  
  if (message.type === 'QUICK_ADD_FROM_SELECTION') {
    chrome.storage.local.set({ selectedText: message.text });
    chrome.action.openPopup();
    sendResponse({ success: true });
  }
  
  return true;
});

// Context menu for right-click "Add to FocusOps"
chrome.runtime.onInstalled.addListener(() => {
  chrome.contextMenus.create({
    id: 'focusops-add-task',
    title: 'Add to FocusOps',
    contexts: ['selection']
  });
  
  chrome.contextMenus.create({
    id: 'focusops-add-link',
    title: 'Add link to FocusOps',
    contexts: ['link']
  });
  
  chrome.contextMenus.create({
    id: 'focusops-add-page',
    title: 'Add page to FocusOps',
    contexts: ['page']
  });
});

// Handle context menu clicks
chrome.contextMenus.onClicked.addListener(async (info, tab) => {
  let taskText = '';
  
  if (info.menuItemId === 'focusops-add-task' && info.selectionText) {
    taskText = info.selectionText;
  } else if (info.menuItemId === 'focusops-add-link' && info.linkUrl) {
    taskText = `Review: ${info.linkUrl}`;
  } else if (info.menuItemId === 'focusops-add-page') {
    taskText = `Review: ${tab?.title || 'Page'} ${tab?.url || ''}`;
  }
  
  if (taskText) {
    await chrome.storage.local.set({ selectedText: taskText });
    chrome.action.openPopup();
  }
});

// Handle extension icon click when popup is open
chrome.action.onClicked.addListener((tab) => {
  // This fires when popup is disabled; popup.html handles normal clicks
  console.log('[FocusOps] Action clicked');
});

console.log('[FocusOps] Background service worker loaded');
